#!/bin/bash

# Firmware Update Script - Linux
# Usage: ./update_firmware.sh

echo "======================================"
echo "     Firmware Update - Linux"
echo "======================================"
echo ""

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Check for dfu-util
echo "Checking for dfu-util..."
if ! command -v dfu-util &> /dev/null; then
    echo "⚠️  dfu-util is not installed."
    echo ""
    echo "Would you like to install it now? (y/n)"
    read -r response
    
    if [[ "$response" =~ ^[Yy]$ ]]; then
        echo "Installing dfu-util..."
        
        # Detect distribution
        if command -v apt-get &> /dev/null; then
            sudo apt-get update
            sudo apt-get install -y dfu-util
        elif command -v dnf &> /dev/null; then
            sudo dnf install -y dfu-util
        elif command -v pacman &> /dev/null; then
            sudo pacman -S --noconfirm dfu-util
        else
            echo "❌ Distribution not recognized. Please install dfu-util manually."
            echo ""
            echo "Press Enter to exit..."
            read -r
            exit 1
        fi
        
        if [ $? -eq 0 ]; then
            echo "✅ dfu-util installed successfully!"
        else
            echo "❌ Installation failed."
            echo ""
            echo "Press Enter to exit..."
            read -r
            exit 1
        fi
    else
        echo "Installation cancelled. Cannot continue without dfu-util."
        echo ""
        echo "Press Enter to exit..."
        read -r
        exit 1
    fi
else
    echo "✅ dfu-util is installed."
fi

echo ""
echo "Searching for .bin files in: $SCRIPT_DIR"
echo ""

# Search for .bin files
BIN_FILES=("$SCRIPT_DIR"/*.bin)

# Check if files were found
if [ ! -e "${BIN_FILES[0]}" ]; then
    echo "❌ No .bin files found in the script directory."
    echo "Please place your firmware file (.bin) next to this script."
    echo ""
    echo "Press Enter to exit..."
    read -r
    exit 1
fi

# Display found files
echo "Firmware files found:"
echo ""
count=0
for file in "${BIN_FILES[@]}"; do
    count=$((count + 1))
    filename=$(basename "$file")
    size=$(du -h "$file" | cut -f1)
    echo "  [$count] $filename ($size)"
done

echo ""
# If only one file, select it automatically
if [ ${#BIN_FILES[@]} -eq 1 ]; then
    SELECTED_FILE="${BIN_FILES[0]}"
    echo "Only one file found, automatically selected."
else
    # Determine most recent file (default)
    latest_index=0
    latest_mtime=0
    for i in "${!BIN_FILES[@]}"; do
        # use stat to get modification time (seconds since epoch)
        mtime=$(stat -c %Y "${BIN_FILES[$i]}" 2>/dev/null || echo 0)
        if [ "$mtime" -gt "$latest_mtime" ]; then
            latest_mtime=$mtime
            latest_index=$i
        fi
    done

    default_choice=$((latest_index + 1))
    echo "Most recent file: [$default_choice] $(basename "${BIN_FILES[$latest_index]}")"
    echo ""
    echo "Press Enter to accept the most recent file, or enter the number of another file (1-$count):"
    read -r selection

    # If user pressed Enter, accept default
    if [ -z "$selection" ]; then
        SELECTED_FILE="${BIN_FILES[$latest_index]}"
    else
        # validate numeric selection
        if ! [[ "$selection" =~ ^[0-9]+$ ]] || [ "$selection" -lt 1 ] || [ "$selection" -gt "$count" ]; then
            echo "❌ Invalid selection."
            echo ""
            echo "Press Enter to exit..."
            read -r
            exit 1
        fi
        SELECTED_FILE="${BIN_FILES[$((selection - 1))]}"
    fi
fi

echo ""
echo "Selected file: $(basename "$SELECTED_FILE")"
echo ""
echo "⚠️  IMPORTANT:"
echo "   1. Connect the kagouyar to your computer via USB"
echo "   2. switch ON the kagouyar"
echo "   3. Put your device in DFU mode: "
echo "       Press and hold the Boot button"
echo "       Press and hold the Reset button"
echo "       Let go of the Reset button"
echo "       Let go of the Boot button"
echo ""
echo "Press Enter to continue or Ctrl+C to cancel..."
read -r

echo ""
echo "Starting firmware update..."
echo "Command: dfu-util -a 0 -s 0x08000000:leave -D '$SELECTED_FILE' -d ,0483:df11"
echo ""

# Execute dfu-util
#dfu-util -a 0 -s 0x08000000:leave -D "$SELECTED_FILE" -d ,0483:df11
OUTPUT=$(dfu-util -a 0 -s 0x08000000:leave -D "$SELECTED_FILE" -d ,0483:df11 2>&1 | tee /dev/tty)

# Afficher la sortie
#echo "$OUTPUT"

# Check result
if echo "$OUTPUT" | grep -q "File downloaded successfully"; then
    echo ""
    echo "✅ Firmware update successful!"
    echo "You can unplug the USB cable"
    echo "and switch OFF the kagouyar for a few seconds"
else
    echo ""
    echo "❌ Update failed."
    echo ""
    echo "Please check that:"
    echo "  - The device is in DFU mode"
    echo "  - The device is connected via USB"
    echo "  - You have the necessary permissions (use sudo if needed)"
fi

echo ""
echo "Press Enter to exit..."
read -r
