# Firmware Update Script - Windows
# Usage: Double-click update_firmware.bat or run in PowerShell

Write-Host "======================================" -ForegroundColor Cyan
Write-Host "     Firmware Update - Windows" -ForegroundColor Cyan
Write-Host "======================================" -ForegroundColor Cyan
Write-Host ""

# Get script directory
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path

# Determine system architecture
$Is64Bit = [Environment]::Is64BitOperatingSystem

# Paths to dfu-util binaries
$dfuUtilWin32 = Join-Path $ScriptDir "dfu-util-0.11-binaries\win32\dfu-util.exe"
$dfuUtilWin64 = Join-Path $ScriptDir "dfu-util-0.11-binaries\win64\dfu-util.exe"

# Check for dfu-util
Write-Host "Checking for dfu-util..." -ForegroundColor Yellow

# First look in system PATH
$dfuUtil = Get-Command dfu-util -ErrorAction SilentlyContinue

if (-not $dfuUtil) {
    # Look in local directories
    if ($Is64Bit -and (Test-Path $dfuUtilWin64)) {
        $dfuUtilPath = $dfuUtilWin64
        Write-Host "✅ dfu-util found in: dfu-util-0.11-binaries\win64\" -ForegroundColor Green
    }
    elseif (Test-Path $dfuUtilWin32) {
        $dfuUtilPath = $dfuUtilWin32
        Write-Host "✅ dfu-util found in: dfu-util-0.11-binaries\win32\" -ForegroundColor Green
    }
    else {
        Write-Host "❌ dfu-util not found." -ForegroundColor Red
        Write-Host ""
        Write-Host "This script looks for dfu-util in:" -ForegroundColor Yellow
        Write-Host "  - $dfuUtilWin32" -ForegroundColor White
        Write-Host "  - $dfuUtilWin64" -ForegroundColor White
        Write-Host ""
        Write-Host "Please ensure the 'dfu-util-0.11-binaries' folder is present" -ForegroundColor Yellow
        Write-Host "next to this script with the following structure:" -ForegroundColor Yellow
        Write-Host "  dfu-util-0.11-binaries\" -ForegroundColor White
        Write-Host "    ├── win32\" -ForegroundColor White
        Write-Host "    │   └── dfu-util.exe" -ForegroundColor White
        Write-Host "    └── win64\" -ForegroundColor White
        Write-Host "        └── dfu-util.exe" -ForegroundColor White
        Write-Host ""
        Read-Host "Press Enter to exit"
        exit 1
    }
}
else {
    $dfuUtilPath = "dfu-util"
    Write-Host "✅ dfu-util is installed in system PATH." -ForegroundColor Green
}

Write-Host ""
Write-Host "Searching for .bin files in: $ScriptDir" -ForegroundColor Yellow
Write-Host ""

# Search for .bin files
$BinFiles = Get-ChildItem -Path $ScriptDir -Filter "*.bin"

# Check if files were found
if ($BinFiles.Count -eq 0) {
    Write-Host "❌ No .bin files found in the script directory." -ForegroundColor Red
    Write-Host "Please place your firmware file (.bin) next to this script." -ForegroundColor Yellow
    Write-Host ""
    Read-Host "Press Enter to exit"
    exit 1
}

# Display found files
Write-Host "Firmware files found:" -ForegroundColor Green
Write-Host ""

$count = 0
foreach ($file in $BinFiles) {
    $count++
    $size = "{0:N2} MB" -f ($file.Length / 1MB)
    Write-Host "  [$count] $($file.Name) ($size)" -ForegroundColor White
}

Write-Host ""

# If only one file, select it automatically
if ($BinFiles.Count -eq 1) {
    $SelectedFile = $BinFiles[0].FullName
    Write-Host "Only one file found, automatically selected." -ForegroundColor Cyan
}
else {
    # Ask user to choose
    do {
        $selection = Read-Host "Enter the number of the file to flash (1-$count)"
        $selectionNum = [int]$selection
    } while ($selectionNum -lt 1 -or $selectionNum -gt $count)
    
    $SelectedFile = $BinFiles[$selectionNum - 1].FullName
}

Write-Host ""
Write-Host "Selected file: $(Split-Path -Leaf $SelectedFile)" -ForegroundColor Green
Write-Host ""
Write-Host "⚠️  IMPORTANT:" -ForegroundColor Yellow
Write-Host "   1. Connect the kagouyar to your computer via USB" -ForegroundColor White
Write-Host "   2. switch on the kagouyar" -ForegroundColor White
Write-Host "   3. Put your device in DFU mode: " -ForegroundColor White
Write-Host "       Press and hold the Boot button" -ForegroundColor White
Write-Host "       Press and hold the Reset button" -ForegroundColor White
Write-Host "       Let go of the Reset button" -ForegroundColor White
Write-Host "       Let go of the Boot button" -ForegroundColor White
Write-Host ""
Read-Host "Press Enter to continue or close the window to cancel"

Write-Host ""
Write-Host "Starting firmware update..." -ForegroundColor Yellow
Write-Host "Command: dfu-util -a 0 -s 0x08000000:leave -D '$SelectedFile' -d ,0483:df11" -ForegroundColor Gray
Write-Host ""

# Execute dfu-util
& $dfuUtilPath -a 0 -s 0x08000000:leave -D $SelectedFile -d ,0483:df11

# Check result
if ($LASTEXITCODE -eq 0) {
    Write-Host ""
    Write-Host "✅ Firmware update successful!" -ForegroundColor Green
    Write-Host "You can unplug the USB cable" -ForegroundColor Green
    Write-Host "and switch off the kagouyar for a few seconds" -ForegroundColor Green
}
else {
    Write-Host ""
    Write-Host "❌ Update failed." -ForegroundColor Red
    Write-Host ""
    Write-Host "Please check that:" -ForegroundColor Yellow
    Write-Host "  - The device is in DFU mode" -ForegroundColor White
    Write-Host "  - The device is connected via USB" -ForegroundColor White
    Write-Host "  - STM32 DFU drivers are installed" -ForegroundColor White
    Write-Host "  - You may need to run the script as administrator" -ForegroundColor White
}

Write-Host ""
Read-Host "Press Enter to exit"
